package mcmas.api;

import java.io.InputStream;
import java.net.URL;

import mcmas.core.MCMCommandQueue;
import mcmas.core.MCMContext;

/**
 * This abstract class provides common methods to all MCMAS plugins.
 * Otherwise, it inherits the same interface as IPlugin.
 *
 * @param <T> the specific type of plugin to return
 */
public abstract class MCMASPlugin<T> implements IPlugin<T> {
	
	private final MCMASContext context;
	
	@Override
	public T newInstance() {
		return newInstance(new MCMASContext());
	}
	
	public MCMASContext getContext() {
		return this.context;
	}
	
	public MCMASPlugin(MCMASContext context) {
		this.context = context;
	}
	
	/**
	 * Get a file URL relative to the current class location.
	 * @param name the name of the file to look for
	 * @return the URL to the file, if it exists
	 */
	public URL getSourceURI(String name) {
		return getClass().getResource(name);
	}
	
	/**
	 * Get a file path relative to the current class location
	 * @param name the name of the file to look for
	 * @return the path of the file as a String, if it exists
	 */
	public String getSourcePath(String name) {
		URL url = getSourceURI(name);
		return (url == null ? null : url.getFile());
	}
	
	/**
	 * Get an InputStream corresponding to a file
	 * relative to the current class location.
	 * @param name the name of the file to look for
	 * @return an InputStream if the file exists, null otherwise
	 */
	public InputStream getSourceStream(String name) {
		return getClass().getResourceAsStream(name);
	}
	
	/**
	 * Get the MCMContext associated to this plugin.
	 * @return a MCM context
	 */
	public MCMContext getMCMContext() {
		return getContext().getContext();
	}
	
	/**
	 * Get the MCMCommandQueue associated to this plugin.
	 * @return a MCM commandQueue
	 */
	public MCMCommandQueue getQueue() {
		return getContext().getQueue();
	}
	
}
